using DicomObjects;
using DicomObjects.Delegates;
using System;
using System.IO;
using System.Threading;
using System.Windows.Forms;

namespace DicomSecurityDemo
{
    public partial class Form1 : Form
    {
        DicomServer server;
        //simple means to access main form for demo program
        public static Form1 MainForm;

        public Form1()
        {
            InitializeComponent();
            MainForm = this;
        }

        private void Form1_Load(object sender, EventArgs e)
        {
            server = new DicomServer();
            server.VerifyReceived += (s, ee) => ee.Status = 0; // return status=0 to verify requests
            ScpChanged(null, null);
        }

        private void Form1_FormClosing(object sender, FormClosingEventArgs e)
        {
            server.UnlistenAll();
        }

        private void ScpChanged(object sender, EventArgs e)
        {
            if (sender is RadioButton rb && rb.Checked)
            {
                server.UnlistenAll();
                Thread.Sleep(1000); // to allow unlisten to complete
                //Only run for the currently checked
                if (rb == WindowsSCP)
                {
                    server.Listen(int.Parse(Form1.MainForm.LocalPort.Text), MinimalWindowsServer.GetDecryptedStream);
                }
                else
                {
                    server.Listen(int.Parse(Form1.MainForm.LocalPort.Text), MinimalBouncyCastleServer.GetDecryptedStream);
                }
            }
        }

        public void ClearResults()
        {
            ServerCert.Text = "";
            ClientCert.Text = "";
            Result.Text = "";
            Refresh();
        }

        private void BouncyCastleSCU_Click(object sender, EventArgs e)
        {
            MinimalBouncyCastleClient BC_Client = new MinimalBouncyCastleClient();
            DoDicomEcho(BC_Client.TlsStream);
        }

        private void WindowsSCU_Click(object sender, EventArgs e)
        {
            // absolute minimum would be :            
            //    DoDicomEcho((stream, address, port)=> new SslStream(stream));
            // but that would not check certificates, nor provide client certs
            MinimalWindowsClient WindowsClient = new MinimalWindowsClient();
            DoDicomEcho(WindowsClient.TlsStream);
        }

        private void DoDicomEcho(TlsInitiator se)
        {
            ClearResults();
            DicomAssociation cn = new DicomAssociation();
            try
            {
                cn.InitiateTls += se;
                cn.Open(RemoteIP.Text, int.Parse(RemotePort.Text), "client", "server");
                var result = cn.Echo();
                Result.Text = $"C-Echo Result : {result}";
                cn.Close();
            }
            catch (Exception ex)
            {
                Result.Text = ex.Message;
            }
        }

        public void SetClientCert(object cert)
        {
            BeginInvoke(new Action(() => ClientCert.Text = (cert == null) ? "None" : cert.ToString()));
        }
        public void SetServerCert(object cert)
        {
            BeginInvoke(new Action(() => ServerCert.Text = (cert == null) ? "None" : cert.ToString()));
        }

        private void Encrypt_Click(object sender, EventArgs e)
        {
            OpenFileDialog ofd = new OpenFileDialog();
            var samplePath = Path.Combine(SampleExtensions.GetSampleImageFolderPath(), SampleExtensions.IMAGE_FOLDER_2D);
            if (Directory.Exists(samplePath))
                ofd.InitialDirectory = samplePath;

            if (ofd.ShowDialog() == DialogResult.OK)
            {
                DicomDataSet dataset = new DicomDataSet(ofd.FileName);

                string encryptedFilename = ofd.FileName + ".encrypted";
                using (FileStream file = new FileStream(encryptedFilename, FileMode.Create))
                using (Stream encryptedStream = BouncyCastleCmsEncryption.Encrypt(file, "password"))
                {
                    dataset.Write(encryptedStream);
                    MessageBox.Show(encryptedFilename, "Output saved");
                }
            }
        }

        private void Decrypt_Click(object sender, EventArgs e)
        {
            OpenFileDialog ofd = new OpenFileDialog();
            var samplePath = Path.Combine(SampleExtensions.GetSampleImageFolderPath(), SampleExtensions.IMAGE_FOLDER_2D);
            if (Directory.Exists(samplePath))
                ofd.InitialDirectory = samplePath;

            if (ofd.ShowDialog() == DialogResult.OK)
            {
                using (FileStream file = new FileStream(ofd.FileName, FileMode.Open))
                using (Stream decryptedStream = BouncyCastleCmsEncryption.Decrypt(file, "password"))
                {
                    DicomDataSet dataset = new DicomDataSet(decryptedStream);
                    string decryptedFilename = ofd.FileName + ".decrypted";
                    dataset.Write(decryptedFilename);
                    MessageBox.Show(decryptedFilename, "Output saved");
                }
            }
        }
    }
}